import numpy as np


def calcu_f(fft_num, sam_rate):
    """
        Return the Discrete Fourier Transform sample frequencies.

        The returned float array `f` contains the frequency bin centers in cycles
        per unit of the sample spacing (with zero at the start).  For instance, if
        the sample spacing is in seconds, then the frequency unit is cycles/second.

        Given a window length `n` and a sample spacing `d`::

        f = [0, 1, ...,   n/2-1,     -n/2, ..., -1] / (d*n)   if n is even
        f = [0, 1, ..., (n-1)/2, -(n-1)/2, ..., -1] / (d*n)   if n is odd

        Parameters
        ----------
        n : int
            Window length.
        d : scalar, optional
            Sample spacing (inverse of the sampling rate). Defaults to 1.

        Returns
        -------
        f : ndarray
            Array of length `n` containing the sample frequencies.
    """

    if fft_num % 2 == 0:
        num = fft_num / 2
        f = np.arange(- num , num) * sam_rate / (fft_num-1)
    else:
        num = (fft_num - 1) / 2
        f = np.arange(- num, num + 1) * sam_rate / (fft_num-1)
    return f
